/* Copyright (C) 2005 Profitability of Hawaii Inc.   All rights reserved.
  
  This software is provided AS-IS with no warranty, either express or
  implied.  See the Alladin Free Public License for details.
  Email: support@poh.com.
  
  This software is distributed under license and may not be copied,
  modified or distributed except as expressly authorized under the terms
  of the license contained in the file LICENSE in this distribution.
  
  For more information about licensing, please refer to
  http://www.ghostscript.com/licensing/. For information on
  commercial licensing, go to http://www.artifex.com/licensing/ or
  contact Artifex Software, Inc., 101 Lucas Valley Road #110,
  San Rafael, CA  94903, U.S.A., +1(415)492-9861.
 */

#include "gdevprn.h"
#include "gdevmem.h"
#include "gdevpccm.h"
#include "gxfont.h"
#include "gscdefs.h"

#include <sys/types.h>
#include <sys/stat.h>

#undef printf
//#define SHOW_PAPER
//#define SHOW_ORIENTATION
//#define SHOW_COLOR_ROTATE_STATUS
//#define SHOW_NEW_COLOR
//#define SHOW_FUNCTION_NAMES
//#define SHOW_CALLBACK_NAMES

#define PNG_INTERNAL
/*
 * libpng versions 1.0.3 and later allow disabling access to the stdxxx
 * files while retaining support for FILE * I/O.
 */
#define PNG_NO_CONSOLE_IO
/*
 * Earlier libpng versions require disabling FILE * I/O altogether.
 * This produces a compiler warning about no prototype for png_init_io.
 * The right thing will happen at link time, since the library itself
 * is compiled with stdio support.  Unfortunately, we can't do this
 * conditionally depending on PNG_LIBPNG_VER, because this is defined
 * in png.h.
 */
/*#define PNG_NO_STDIO*/
#include "png_.h"

/* ------ The device descriptors ------ */

/*
 * Default X and Y resolution.
 */
#define X_DPI 72
#define Y_DPI 72

static dev_proc_get_params(pngbest_get_params);
static dev_proc_put_params(pngbest_put_params);
static dev_proc_encode_color(pngbest_encode_color);
static dev_proc_decode_color(pngbest_decode_color);
static dev_proc_open_device(pngbest_open_device);
static dev_proc_close_device(pngbest_close_device);
static dev_proc_print_page(pngbest_print_page);
static dev_proc_output_page(pngbest_prn_output_page);
static dev_proc_get_bits_rectangle(pngbest_get_bits_rectangle);
static dev_proc_fill_rectangle(pngbest_fill_rectangle);
static dev_proc_copy_mono(pngbest_copy_mono);
static dev_proc_copy_color(pngbest_copy_color);
static dev_proc_fill_trapezoid(pngbest_fill_trapezoid);
static dev_proc_fill_parallelogram(pngbest_fill_parallelogram);
static dev_proc_fill_triangle(pngbest_fill_triangle);
static dev_proc_draw_thin_line(pngbest_draw_thin_line);
static dev_proc_begin_image(pngbest_begin_image);
static dev_proc_strip_tile_rectangle(pngbest_strip_tile_rectangle);
static dev_proc_strip_copy_rop(pngbest_strip_copy_rop);
static dev_proc_begin_typed_image(pngbest_begin_typed_image);
static dev_proc_create_compositor(pngbest_create_compositor);
static dev_proc_text_begin(pngbest_text_begin);

//local utility funcitons
inline int pngbest_str_binary_search(char * searchStr, char ** strArray, int numElements);

typedef struct color_cube_s color_cube;
struct color_cube_s
{
    byte index;
    color_cube *next;
};

typedef struct pngbest_page_dsc_info_s
{
    int orientation;		/* -1 .. 3 */
    int viewing_orientation;	/* -1 .. 3 */
} pngbest_page_dsc_info_t;

typedef struct gx_device_pngbest_s {
    gx_device_common;
    gx_prn_device_common;
    color_cube *cube;
    png_color *palette;
    int color_count;
    int color_count_is_accurate;
    int rotated;
    int itextflag;
    pngbest_page_dsc_info_t doc_dsc_info; /* document default */
    pngbest_page_dsc_info_t page_dsc_info; /* current page */ 
    dev_t_proc_fill_rectangle((*orig_fill_rectangle), gx_device);
    dev_t_proc_copy_mono((*orig_copy_mono), gx_device);
    dev_t_proc_copy_color((*orig_copy_color), gx_device);
    dev_t_proc_fill_trapezoid((*orig_fill_trapezoid), gx_device);
    dev_t_proc_fill_parallelogram((*orig_fill_parallelogram), gx_device);
    dev_t_proc_fill_triangle((*orig_fill_triangle), gx_device);
    dev_t_proc_draw_thin_line((*orig_draw_thin_line), gx_device);
    dev_t_proc_begin_image((*orig_begin_image), gx_device);
    dev_t_proc_strip_tile_rectangle((*orig_strip_tile_rectangle), gx_device);
    dev_t_proc_strip_copy_rop((*orig_strip_copy_rop), gx_device);
    dev_t_proc_begin_typed_image((*orig_begin_typed_image), gx_device);
    dev_t_proc_create_compositor((*orig_create_compositor), gx_device);
} gx_device_pngbest;

static const gx_device_procs pngbest_procs =
{
    pngbest_open_device,
    NULL, /* get_initial_matrix */
    NULL, /* sync_output */
    gdev_prn_output_page,
    pngbest_close_device,
    NULL, // map_rgb_color ==> obsolete
    NULL, // map_color_rgb ==> obsolete
    NULL, ///////////////// pngbest_fill_rectangle,
    NULL, // tile_rectangle ==> obsolete
    NULL, ///////////////// pngbest_copy_mono,
    NULL, ///////////////// pngbest_copy_color,
    NULL, // draw_line ==> obsolete
    NULL, /* get_bits */
    pngbest_get_params,
    pngbest_put_params,
    NULL, // map_cmyk_color ==> obolete
    NULL, /* get_xfont_procs */
    NULL, /* get_xfont_device */
    NULL, /* map_rgb_alpha_color */
    gx_page_device_get_page_device,
    NULL, // get_alpha_bits ==> obsolete
    NULL, /*pngalpha_copy_alpha,*/
    NULL, /* get_band */
    NULL, /* copy_rop */ // ==> strip_copy_rop
    NULL, /* fill_path */   // ==> high level ------------------
    NULL, /* stroke_path */ // ==> high level ------------------
    NULL, /* fill_mask */   // ==> high level ------------------
    NULL, ///////////////// pngbest_fill_trapezoid,
    NULL, ///////////////// pngbest_fill_parallelogram,
    NULL, ///////////////// pngbest_fill_triangle,
    NULL, ///////////////// pngbest_draw_thin_line,
    NULL, ///////////////// pngbest_begin_image,
    NULL, /* image_data */
    NULL, /* end_image */
    NULL, ///////////////// pngbest_strip_tile_rectangle,
    NULL, ///////////////// pngbest_strip_copy_rop,
    NULL, /* get_clipping_box */
    NULL, ///////////////// pngbest_begin_typed_image,
    NULL, /* get_bits_rectangle */
    NULL, /* map_color_rgb_alpha */
    NULL, ///////////////// pngbest_create_compositor,
    NULL, /* get_hardware_params */
    pngbest_text_begin,
    NULL, /* finish_copydevice */
    NULL, /* begin_transparency_group */
    NULL, /* end_transparency_group */
    NULL, /* begin_transparency_mask */
    NULL, /* end_transparency_mask */
    NULL, /* discard_transparency_layer */
    gx_default_DevRGB_get_color_mapping_procs,
    gx_default_DevRGB_get_color_comp_index, 
    pngbest_encode_color,
    pngbest_decode_color
};

const gx_device_pngbest gs_pngbest_device = {
    std_device_part1_(gx_device_pngbest, &pngbest_procs, "pngbest",
    &st_device_printer, open_init_closed),
    /* color_info */ 
    {3 /* max components */,
     3 /* number components */,
     GX_CINFO_POLARITY_ADDITIVE /* polarity */,
     24 /* depth */,
     -1 /* gray index */,
     255 /* max gray */,
     255 /* max color */,
     256 /* dither grays */,
     256 /* dither colors */,
    { 1, 1 } /* antialias info text, graphics */,
     GX_CINFO_SEP_LIN_NONE /* separable_and_linear */,
     { 0 } /* component shift */,
     { 0 } /* component bits */,
     { 0 } /* component mask */,
     "DeviceRGB" /* process color name */,
     GX_CINFO_OPMODE_UNKNOWN /* opmode */,
     0 /* process_cmps */
    },
    std_device_part2_(
      (int)((float)(DEFAULT_WIDTH_10THS) * (X_DPI) / 10 + 0.5),
      (int)((float)(DEFAULT_HEIGHT_10THS) * (Y_DPI) / 10 + 0.5),
       X_DPI, Y_DPI),
    offset_margin_values(0, 0, 0, 0, 0, 0),
    std_device_part3_(),
    prn_device_body_rest_(pngbest_print_page),
    NULL, /* color_cube *cube */
    NULL, /* png_color *palette */
    0,    /* int color_count */
    0,    /* int color_count_is_accurate */
    -1,   /* rotated */
    0,    /* itext */
    { 0 }, /* doc info */
    { 0 }, /* page info */
    NULL, /* fill_rectangle */
    NULL, /* copy_mono */
    NULL, /* copy_color */
    NULL, /* fill_trapezoid */
    NULL, /* fill_parallelogram */
    NULL, /* fill_triangle */
    NULL, /* draw_thin_line */
    NULL, /* begin_image */
    NULL, /* strip_tile_rectangle */
    NULL, /* strip_copy_rop */
    NULL, /* begin_typed_image */
    NULL, /* create_compositor */
};

/* --- debug functions --- */

#if 0
static void 
show_bits(char *text, ulong number)
{
    int i = 0;
    ulong mask = 0x80000000;

    if ( text )
        printf(text);

    for ( ; i<32; i++ )
    {
        printf("%d", (number & mask) ? 1 : 0);
        mask = mask >> 1;
    }

    printf("\n");
}

static void
pngbest_dump_palette(gx_device_pngbest *pngbestdev)
{
    int i;

    printf("\n");

    for ( i=0; i<pngbestdev->color_count; i++ )
    {
        printf("RGB = [%02x %02x %02x]\n", pngbestdev->palette[i].red,
                                           pngbestdev->palette[i].green,
                                           pngbestdev->palette[i].blue);
    }

    printf("\n");
}
#endif

/* ------ static definitions ------ */

static void
pngbest_dsc_process(gx_device_pngbest * pngbestdev, const gs_param_string_array * pma)
{
    int i;

#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_dsc_process()...\n");
#endif

    for ( i=0; i+1 < pma->size; i += 2 )
    {
        const gs_param_string *pkey = &pma->data[i];
        const gs_param_string *pvalue = &pma->data[i + 1];

        pngbest_page_dsc_info_t *ppdi;

        if ( (ppdi = &pngbestdev->doc_dsc_info, pdf_key_eq(pkey, "Orientation")) ||
             (ppdi = &pngbestdev->page_dsc_info, pdf_key_eq(pkey, "PageOrientation")) ) 
        {
            if (pvalue->size == 1 && pvalue->data[0] >= '0' && pvalue->data[0] <= '3' )
                ppdi->orientation = pvalue->data[0] - '0';
            else
                ppdi->orientation = -1;

#ifdef SHOW_ORIENTATION
            printf("%s: %d\n", (pdf_key_eq(pkey, "Orientation") ? "Orientation" : "PageOrientation") , ppdi->orientation);
#endif
        } 
        else if ( (ppdi = &pngbestdev->doc_dsc_info, pdf_key_eq(pkey, "ViewingOrientation")) || 
                  (ppdi = &pngbestdev->page_dsc_info, pdf_key_eq(pkey, "PageViewingOrientation")) )
        {
            gs_matrix mat;
            int orient;

            if ( sscanf((const char *)pvalue->data, "[%g %g %g %g]", &mat.xx, &mat.xy, &mat.yx, &mat.yy) != 4 )
                continue; /* error */
            for (orient = 0; orient < 4; ++orient) 
            {
                if (mat.xx == 1 && mat.xy == 0 && mat.yx == 0 && mat.yy == 1)
                    break;
                gs_matrix_rotate(&mat, -90.0, &mat);
            }
            if (orient == 4) /* error */
                orient = -1;

            ppdi->viewing_orientation = orient;

#ifdef SHOW_ORIENTATION
            printf("%s: %d\n", (pdf_key_eq(pkey, "ViewingOrientation") ? "ViewingOrientation" : "PageViewingOrientation") , ppdi->viewing_orientation);
#endif
        }
    }
}

int
pngbest_put_params(gx_device *dev, gs_param_list *plist)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;
    int code;
    gs_param_string_array ppa;
    int temp;

#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_put_params()...\n");
#endif

    code = param_read_int(plist, "ITEXT", &temp);
    switch (code) 
    {
        case 0:
            pngbestdev->itextflag=temp;
            break;
        case 1:/* not found */
            code = 0;
            break;
        default:
            break;
    }

    if ( code == 0 )
    {
        code = param_read_string_array(plist, "DSC", &ppa);
        switch (code) 
        {
        case 0:
            pngbest_dsc_process(pngbestdev, &ppa);
            break;
        case 1:		/* not found */
            code = 0;
            break;
        default:
            param_signal_error(plist, "DSC", code);
            break;
        }
    }

    if ( code == 0 )
    {
        code = gdev_prn_put_params(dev, plist);
    }

    return code;
}

/* Get device parameters */
static int
pngbest_get_params(gx_device *pdev, gs_param_list *plist)
{
    gx_device_pngbest *ppdev = (gx_device_pngbest *)pdev;
    int code;
    
#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_get_params()...\n");
#endif

    code = gdev_prn_get_params(pdev, plist);
    if (code >= 0)
        code = param_write_int(plist, "ITEXT", &(ppdev->itextflag));

    return code;
}

static gx_color_index
pngbest_encode_color(gx_device * dev, const gx_color_value cv[])
{
    return gx_color_value_to_byte(cv[2]) +
           ((uint) gx_color_value_to_byte(cv[1]) << 8) +
           ((ulong) gx_color_value_to_byte(cv[0]) << 16);
}

static int
pngbest_decode_color(gx_device * dev, gx_color_index color,
                 gx_color_value prgb[3])
{
    prgb[0] = gx_color_value_from_byte(color >> 16);
    prgb[1] = gx_color_value_from_byte((color >> 8) & 0xff);
    prgb[2] = gx_color_value_from_byte(color & 0xff);

    return 0;
}

/* return 0 good */
/*        1 bad  */
static int
pngbest_add_color(gx_device_pngbest *pngbestdev, png_color *c)
{
    /* create color cube */

    if ( pngbestdev->cube[c->red].next == NULL ) /* create green row */
    {
        pngbestdev->cube[c->red].next = (color_cube *)malloc(256*sizeof(color_cube));
        if (pngbestdev->cube[c->red].next == 0)
            return 1;
        memset(pngbestdev->cube[c->red].next, 0, 256*sizeof(color_cube));
    }

    if ( pngbestdev->cube[c->red].next[c->green].next == NULL ) /* create blue row */
    {
        pngbestdev->cube[c->red].next[c->green].next = (color_cube *)malloc(256*sizeof(color_cube));
        if (pngbestdev->cube[c->red].next[c->green].next == 0)
            return 1;
        memset(pngbestdev->cube[c->red].next[c->green].next, 0, 256*sizeof(color_cube));
    }

    if ( !pngbestdev->cube[c->red].next[c->green].next[c->blue].next )
    {
        /* mark color as used */
        pngbestdev->cube[c->red].next[c->green].next[c->blue].index = pngbestdev->color_count; /* palette index */
        pngbestdev->cube[c->red].next[c->green].next[c->blue].next = (color_cube *)1; /* mark color as used */

#ifdef SHOW_NEW_COLOR
        printf("add new RGB = [%02x %02x %02x] -> palette index = %d\n", c->red,
                                                                         c->green,
                                                                         c->blue,
                                                                         pngbestdev->color_count);
#endif
        /* add new color to palette */
        if ( pngbestdev->color_count < 256 )
        {
            pngbestdev->palette[pngbestdev->color_count].red   = c->red;
            pngbestdev->palette[pngbestdev->color_count].green = c->green;
            pngbestdev->palette[pngbestdev->color_count].blue  = c->blue;
        }

        pngbestdev->color_count++;
    }

    return 0;
}

#if 0
static int
pngbest_is_mono(gx_device_pngbest *pngbestdev)
{
    int i;

    for ( i=0; i<pngbestdev->color_count; i++ )
    {
        if ( !( (pngbestdev->palette[i].red   == 0x00 &&
                 pngbestdev->palette[i].green == 0x00 &&
                 pngbestdev->palette[i].blue  == 0x00) ||
                (pngbestdev->palette[i].red   == 0xff &&
                 pngbestdev->palette[i].green == 0xff &&
                 pngbestdev->palette[i].blue  == 0xff)    ) )
        {
            return 0;
        }
    }

    return 1;
}
#endif

static int
pngbest_is_grayscale(gx_device_pngbest *pngbestdev)
{
    int i;

    for ( i=0; i<pngbestdev->color_count; i++ )
    {
        if ( pngbestdev->palette[i].red   != pngbestdev->palette[i].green || 
             pngbestdev->palette[i].green != pngbestdev->palette[i].blue     )
        {
            return 0;
        }
    }

    return 1;
}

static void
pngbest_check_rotation(gx_device_pngbest *pngbestdev)
{
    char papersize[50];

#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_check_rotation()...\n");
#endif

    if ( pngbestdev->rotated != -1 )
        return;

    /* to keep it simple for now */
    if ( pngbestdev->doc_dsc_info.orientation          > 0 ||
         pngbestdev->page_dsc_info.orientation         > 0 ||
         pngbestdev->doc_dsc_info.viewing_orientation  > 0 ||
         pngbestdev->page_dsc_info.viewing_orientation > 0    )
    {
        pngbestdev->rotated = 1;
    }
    else
    {
        pngbestdev->rotated = 0;
    }
}

static void
pngbest_init_color_cube(gx_device_pngbest *pngbestdev)
{
    memset(pngbestdev->cube, 0, 256 * sizeof(color_cube));
    memset(pngbestdev->palette, 0, 256 * sizeof(png_color));
    pngbestdev->color_count = 0;
}

static void
pngbest_delete_color_cube(gx_device_pngbest *pngbestdev)
{
    int x, y;

    for ( x=0; x<256; x++ )
    {
        if ( pngbestdev->cube[x].next )
        {
            for ( y=0; y<256; y++ )
            {
                if ( pngbestdev->cube[x].next[y].next )
                {
                    /* free blue rows */
                    free(pngbestdev->cube[x].next[y].next); 
                }
            }
            /* free green rows */
            free(pngbestdev->cube[x].next);
        }
    }
}

static void
pngbest_start_page(gx_device_pngbest *pngbestdev)
{
#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_start_page() ...\n");
#endif

    pngbestdev->color_count_is_accurate = 1;

    pngbest_init_color_cube(pngbestdev);

    // reset the "per page" rotation values
    pngbestdev->rotated = -1;
    pngbestdev->page_dsc_info.orientation = -1;
    pngbestdev->page_dsc_info.viewing_orientation = -1;

    if ((pngbestdev->procs.fill_rectangle != pngbest_fill_rectangle) &&	(pngbestdev->procs.fill_rectangle != NULL)) 
    {
        pngbestdev->orig_fill_rectangle = pngbestdev->procs.fill_rectangle;
        pngbestdev->procs.fill_rectangle = pngbest_fill_rectangle;
    }

    if ((pngbestdev->procs.copy_mono != pngbest_copy_mono) && (pngbestdev->procs.copy_mono != NULL)) 
    {
        pngbestdev->orig_copy_mono = pngbestdev->procs.copy_mono;
        pngbestdev->procs.copy_mono = pngbest_copy_mono;
    }

    if ((pngbestdev->procs.copy_color != pngbest_copy_color) && (pngbestdev->procs.copy_color != NULL)) 
    {
        pngbestdev->orig_copy_color = pngbestdev->procs.copy_color;
        pngbestdev->procs.copy_color = pngbest_copy_color;
    }

    if ((pngbestdev->procs.fill_trapezoid != pngbest_fill_trapezoid) && (pngbestdev->procs.fill_trapezoid != NULL)) 
    {
        pngbestdev->orig_fill_trapezoid = pngbestdev->procs.fill_trapezoid;
        pngbestdev->procs.fill_trapezoid = pngbest_fill_trapezoid;
    }

    if ((pngbestdev->procs.fill_parallelogram != pngbest_fill_parallelogram) && (pngbestdev->procs.fill_parallelogram != NULL)) 
    {
        pngbestdev->orig_fill_parallelogram = pngbestdev->procs.fill_parallelogram;
        pngbestdev->procs.fill_parallelogram = pngbest_fill_parallelogram;
    }

    if ((pngbestdev->procs.fill_triangle != pngbest_fill_triangle) && (pngbestdev->procs.fill_triangle != NULL)) 
    {
        pngbestdev->orig_fill_triangle = pngbestdev->procs.fill_triangle;
        pngbestdev->procs.fill_triangle = pngbest_fill_triangle;
    }

    if ((pngbestdev->procs.draw_thin_line != pngbest_draw_thin_line) && (pngbestdev->procs.draw_thin_line != NULL)) 
    {
        pngbestdev->orig_draw_thin_line = pngbestdev->procs.draw_thin_line;
        pngbestdev->procs.draw_thin_line = pngbest_draw_thin_line;
    }

    if ((pngbestdev->procs.begin_image != pngbest_begin_image) && (pngbestdev->procs.begin_image != NULL)) 
    {
        pngbestdev->orig_begin_image = pngbestdev->procs.begin_image;
        pngbestdev->procs.begin_image = pngbest_begin_image;
    }

    if ((pngbestdev->procs.strip_tile_rectangle != pngbest_strip_tile_rectangle) && (pngbestdev->procs.strip_tile_rectangle != NULL)) 
    {
        pngbestdev->orig_strip_tile_rectangle = pngbestdev->procs.strip_tile_rectangle;
        pngbestdev->procs.strip_tile_rectangle = pngbest_strip_tile_rectangle;
    }

    if ((pngbestdev->procs.strip_copy_rop != pngbest_strip_copy_rop) && (pngbestdev->procs.strip_copy_rop != NULL)) 
    {
        pngbestdev->orig_strip_copy_rop = pngbestdev->procs.strip_copy_rop;
        pngbestdev->procs.strip_copy_rop = pngbest_strip_copy_rop;
    }

    if ((pngbestdev->procs.begin_typed_image != pngbest_begin_typed_image) && (pngbestdev->procs.begin_typed_image != NULL)) 
    {
        pngbestdev->orig_begin_typed_image = pngbestdev->procs.begin_typed_image;
        pngbestdev->procs.begin_typed_image = pngbest_begin_typed_image;
    }

    if ((pngbestdev->procs.create_compositor != pngbest_create_compositor) && (pngbestdev->procs.create_compositor != NULL)) 
    {
        pngbestdev->orig_create_compositor = pngbestdev->procs.create_compositor;
        pngbestdev->procs.create_compositor = pngbest_create_compositor;
    }
}

static void
pngbest_finish_page(gx_device_pngbest *pngbestdev)
{
#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_finish_page() ...\n\n");
#endif

    pngbest_delete_color_cube(pngbestdev);
}

static int
pngbest_open_device(gx_device * dev)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;
    int code;

#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_open_device() ...\n");
#endif

    code = gdev_prn_open(dev);

    pngbestdev->cube = (color_cube *)malloc(256 * sizeof(color_cube));
    pngbestdev->palette = (png_color *)malloc(256 * sizeof(png_color));
    
    if (pngbestdev->cube == 0 || pngbestdev->palette == 0)
        return gs_note_error(gs_error_VMerror);

    // the "global" rotation values are reset only once
    pngbestdev->doc_dsc_info.orientation = -1;
    pngbestdev->doc_dsc_info.viewing_orientation = -1;

    /* init color cube and palette */
    pngbest_start_page(pngbestdev);

    return code;
}

static int
pngbest_close_device(gx_device * dev)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;
    int code;

#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_close_device() ...\n");
#endif
   
    /* delete color cube and palette */
    pngbest_finish_page(pngbestdev);

    free(pngbestdev->cube);
    free(pngbestdev->palette);

    code = gdev_prn_close(dev);

    return code;
}

pngbest_prn_output_page(gx_device * dev, int num_copies, int flush)
{
    gx_device_printer * const pdev = (gx_device_printer *)dev;
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;
    int code;

#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_prn_output_page() ...\n");
#endif

//    code = (*pdev->printer_procs.print_page)(pdev, NULL);
//    if (code < 0)
//        return code;

    code = (pdev->buffer_space && !pdev->is_async_renderer ?
            clist_finish_page(dev, flush) : 0);
    if (code < 0)
        return code;

    return gx_finish_output_page(dev, num_copies, flush);
}

/* Write out a page in PNG format. */
/* This routine is used for all formats. */
static int
pngbest_print_page(gx_device_printer * pdev, FILE * file)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)pdev;
    gs_memory_t *mem = pdev->memory;
    int raster = pdev->width * 3;
    int x, width = pdev->width;
    int y, height = pdev->height;
    int i, j, k;
    png_color c;
    byte *buffer = NULL; /* big image buffer */
    byte *row = NULL; /* png scanline buffer */
    byte *p;
    byte bits;
    int code; /* return code */

    /* PNG structures */
    png_struct *png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
    png_info *info_ptr = png_create_info_struct(png_ptr);
    char software_key[80];
    char software_text[256];
    png_text text_png;

#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_print_page() ...\n");
#endif
    
    /* set error handling */
    if (setjmp(png_ptr->jmpbuf)) {
        /* If we get here, we had a problem reading the file */
        code = gs_note_error(gs_error_VMerror);
        goto done;
    }

    code = 0; /* for normal path */

    if ( pngbestdev->rotated == -1 )
        pngbest_check_rotation(pngbestdev);

    pngbestdev->color_count_is_accurate = 0;

    if ( !pngbestdev->color_count_is_accurate || pngbestdev->rotated )
    {
        if ( !pngbestdev->color_count_is_accurate )
        {
#ifdef SHOW_COLOR_ROTATE_STATUS
            printf("!!! count colors !!!\n");
#endif
            pngbest_delete_color_cube(pngbestdev); /* destroy current color cube */
            pngbest_init_color_cube(pngbestdev);  /* init new color cube and palette */
        }

#ifdef SHOW_COLOR_ROTATE_STATUS
        if ( pngbestdev->rotated ) printf("!!! rotate !!!\n");
#endif

        /* allocate buffer for entire image */
        buffer = gs_alloc_bytes(mem, width * height * 3, "png raster buffer");
        if ( buffer == 0 )
        {
            code = gs_note_error(gs_error_VMerror);
            goto done;
        }

        for ( y=0; y<height; y++ )
        {
            gdev_prn_get_bits(pdev, y, NULL, &p);

            if ( !pngbestdev->color_count_is_accurate && pngbestdev->color_count <= 256 )
            {
                for ( x=0; x<raster; )
                {
                    c.red   = p[x++];
                    c.green = p[x++];
                    c.blue  = p[x++];

                    if ( pngbest_add_color(pngbestdev, &c) )
                    {
                        code = gs_note_error(gs_error_VMerror);
                        goto done;
                    }
                }
            }

            if ( pngbestdev->rotated )
            {
                /* rotate scanline */
                for ( x=0, i=0; x<raster; i++ )
                {
                    int offset = (i * height * 3) + ((height-y-1) * 3);

                    buffer[offset    ] = p[x++];
                    buffer[offset + 1] = p[x++];
                    buffer[offset + 2] = p[x++];
                }
            }
            else
            {
                memcpy(buffer + (y * raster), p, raster);
            }
        }
    }

    if ( pngbestdev->rotated )
    {
        width = pdev->height;
        height = pdev->width;
        raster = width * 3;
    }

    /* set up the output control */
    png_init_io(png_ptr, file);

    /* set the file information here */
    info_ptr->width = width;
    info_ptr->height = height;

    /* resolution is in pixels per meter vs. dpi */
    info_ptr->x_pixels_per_unit = (png_uint_32) (pdev->HWResolution[0] * (100.0 / 2.54));
    info_ptr->y_pixels_per_unit = (png_uint_32) (pdev->HWResolution[1] * (100.0 / 2.54));
    info_ptr->phys_unit_type = PNG_RESOLUTION_METER;
    info_ptr->valid |= PNG_INFO_pHYs;

    if ( pngbestdev->color_count <= 2 )
    {
#ifdef SHOW_COLOR_ROTATE_STATUS
        printf("number of colors: %d => 1 bit (black and white)\n", pngbestdev->color_count);
#endif
        row = gs_alloc_bytes(mem, width/8+1, "png scanline buffer");

        info_ptr->bit_depth = 1;
        info_ptr->color_type = PNG_COLOR_TYPE_PALETTE;
        pngbestdev->color_count = 2;
    }
    else if ( pngbestdev->color_count <= 16 )
    {
#ifdef SHOW_COLOR_ROTATE_STATUS
        printf("number of colors: %d => 4 bit (palette)\n", pngbestdev->color_count);
#endif
//        pngbest_dump_palette(info_ptr->palette, color_count);

        row = gs_alloc_bytes(mem, width/2+1, "png scanline buffer");

        info_ptr->bit_depth = 4;
        info_ptr->color_type = PNG_COLOR_TYPE_PALETTE;
        pngbestdev->color_count = 16;
    }
    else if ( pngbestdev->color_count <= 256 )
    {
#ifdef SHOW_COLOR_ROTATE_STATUS
        printf("number of colors: %d => 8 bit ", pngbestdev->color_count);
#endif
        info_ptr->bit_depth = 8;
        pngbestdev->color_count = 256;

        row = gs_alloc_bytes(mem, width, "png scanline buffer");

        if ( pngbest_is_grayscale(pngbestdev) )
        {
#ifdef SHOW_COLOR_ROTATE_STATUS
            printf("(gray)\n");
#endif
            info_ptr->color_type = PNG_COLOR_TYPE_GRAY;
        }
        else
        {
#ifdef SHOW_COLOR_ROTATE_STATUS
            printf("(palette)\n");
#endif
            info_ptr->color_type = PNG_COLOR_TYPE_PALETTE;
        }

//        pngbest_dump_palette(info_ptr->palette, color_count);
    }
    else 
    {
#ifdef SHOW_COLOR_ROTATE_STATUS
        printf("number of colors: %d => use 24 bits\n", pngbestdev->color_count);
#endif

        info_ptr->bit_depth = 8;
        info_ptr->color_type = PNG_COLOR_TYPE_RGB;
    }

    /* activate the palette if there is one */
    if (info_ptr->color_type == PNG_COLOR_TYPE_PALETTE) {
        
        info_ptr->palette = pngbestdev->palette;
        info_ptr->num_palette = pngbestdev->color_count; /* 2 or 16 or 256 colors */
        info_ptr->valid |= PNG_INFO_PLTE;
    }

    /* add comment */
    strncpy(software_key, "Software", sizeof(software_key));
    sprintf(software_text, "%s %d.%02d", gs_product,
            (int)(gs_revision / 100), (int)(gs_revision % 100));
    text_png.compression = -1;	/* uncompressed */
    text_png.key = software_key;
    text_png.text = software_text;
    text_png.text_length = strlen(software_text);
    info_ptr->text = &text_png;
    info_ptr->num_text = 1;

    /* write the file information */
    png_write_info(png_ptr, info_ptr);

    /* don't write the comments twice */
    info_ptr->num_text = 0;
    info_ptr->text = NULL;

    /* Write the contents of the image. */
    if (info_ptr->color_type == PNG_COLOR_TYPE_PALETTE)
    {
        for (y = 0; y < height; y++)
        {
            if ( pngbestdev->color_count_is_accurate && !pngbestdev->rotated )
                gdev_prn_get_bits(pdev, y, NULL, &p);
            else 
                p = buffer + (y * raster);

            for ( x=0, i=0; x<raster; i++ )
            {
                for ( j=0; j<8 && x<raster; j+=info_ptr->bit_depth)
                {
                    c.red   = p[x++];
                    c.green = p[x++];
                    c.blue  = p[x++];

                    if (pngbestdev->cube[c.red].next != NULL &&
                        pngbestdev->cube[c.red].next[c.green].next != NULL)
                        bits = pngbestdev->cube[c.red].next[c.green].next[c.blue].index << ((8-info_ptr->bit_depth)-j);
                    else
                        bits = 0;

                    if (j==0)
                        row[i] = bits;
                    else
                        row[i] |= bits;
                }
            }

            png_write_rows(png_ptr, &row, 1);
        }
    } else
    {
        switch ( pngbestdev->color_count )
        {
            case 256: /* 8 bit */
            {
                if ( info_ptr->color_type == PNG_COLOR_TYPE_GRAY )
                {
                    for (y = 0; y < height; y++)
                    {
                        if ( pngbestdev->color_count_is_accurate && !pngbestdev->rotated )
                            gdev_prn_get_bits(pdev, y, NULL, &p);
                        else 
                            p = buffer + (y * raster);
    
                        for ( x=0, i=0; x<raster; x+=3, i++ )
                        {
                            row[i] = p[x];
                        }
    
                        png_write_rows(png_ptr, &row, 1);
                    }
                }
		break;
    
            }
            default: /* 24 bit */
            {
                for (y = 0; y < height; y++) 
                {
                    if ( pngbestdev->color_count_is_accurate && !pngbestdev->rotated )
                        gdev_prn_get_bits(pdev, y, NULL, &p);
                    else 
                        p = buffer + (y * raster);
    
                    png_write_rows(png_ptr, &p, 1);
                }
            }
        }
    }

    /* write the rest of the file */
    png_write_end(png_ptr, info_ptr);

done:
    /* free the structures */
    png_destroy_write_struct(&png_ptr, &info_ptr);

    /* free the scanline buffer */
    gs_free_object(mem, row, "png scanline buffer");

    /* free the big raster buffer */
    gs_free_object(mem, buffer, "png raster buffer");

    /* destroy color cube and palette */
    pngbest_finish_page(pngbestdev);

    /* prepare for new page */
    pngbest_start_page(pngbestdev);

    return code;
}

static int
pngbest_capture_color(gx_device * dev, gx_color_index color)
{
    gx_color_value rgb[3];
    png_color c;

#ifdef SHOW_FUNCTION_NAMES
    printf("pngbest_capture_color()...\n");
#endif

    (*dev_proc((gx_device_printer *)dev, decode_color)) (dev, color, rgb);

    c.red   = gx_color_value_to_byte(rgb[0]);
    c.green = gx_color_value_to_byte(rgb[1]);
    c.blue  = gx_color_value_to_byte(rgb[2]);

    return pngbest_add_color((gx_device_pngbest *)dev, &c);
}

/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_fill_rectangle(gx_device * dev, int x, int y, int w, int h,
		  gx_color_index color)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;
    int code;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_fill_rectangle() ...\n");
#endif    

    if ( pngbestdev->color_count_is_accurate && pngbestdev->color_count <= 256 )
    {
        if ( pngbest_capture_color(dev, color) )
        {
            code = gs_note_error(gs_error_VMerror);
            goto done;
        }
    }
    else
    {
        /* unregister this callback function */
        dev->procs.fill_rectangle = pngbestdev->orig_fill_rectangle;
    }

    code = pngbestdev->orig_fill_rectangle(dev, x, y, w, h, color);
   
done:
    return code;
}

/* --------------------------------------------------------------------------*/
/* --- not supported --------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int 
pngbest_copy_mono(gx_device *dev, const unsigned char *data, int data_x, 
              int raster, gx_bitmap_id id, int x, int y, 
              int width, int height, 
              gx_color_index color0, gx_color_index color1)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_copy_mono() ...\n");
#endif    

    /* we don't know the outcome of this operation */
    pngbestdev->color_count_is_accurate = 0;

    /* unregister this callback function */
    dev->procs.copy_mono = pngbestdev->orig_copy_mono;

#ifdef SHOW_CALLBACK_NAMES
    printf("\n\n!!!\n\n");
#endif

    return dev->procs.copy_mono(dev, data, data_x, raster, id, x, y, 
                                width, height, color0, color1);
}

/* --------------------------------------------------------------------------*/
/* --- not supported --------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_copy_color(gx_device *dev, const unsigned char *data, int data_x,
               int raster, gx_bitmap_id id, int x, int y, 
               int width, int height)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_copy_color() ...\n");
#endif    

    /* we don't know the outcome of this operation */
    pngbestdev->color_count_is_accurate = 0;

    /* unregister this callback function */
    dev->procs.copy_color = pngbestdev->orig_copy_color;

#ifdef SHOW_CALLBACK_NAMES
    printf("\n\n!!!\n\n");
#endif

    return dev->procs.copy_color(dev, data, data_x, raster, id, x, y, 
                                 width, height);
}

/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_fill_trapezoid(gx_device *dev, const gs_fixed_edge *left, 
                   const gs_fixed_edge *right, fixed ybot, fixed ytop, 
                   bool swap_axes, const gx_drawing_color *pdcolor, 
                   gs_logical_operation_t lop)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;
    int code;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_fill_trapezoid() ...\n");
#endif    

    if ( pngbestdev->color_count_is_accurate && pngbestdev->color_count <= 256 )
    {
        if ( pngbest_capture_color(dev, pdcolor->colors.pure) )
        {
            code = gs_note_error(gs_error_VMerror);
            goto done;
        }
    }
    else
    {
        /* unregister this callback function */
        dev->procs.fill_trapezoid = pngbestdev->orig_fill_trapezoid;
    }

    code = pngbestdev->orig_fill_trapezoid(dev, left, right, ybot, ytop, swap_axes, pdcolor, lop);
   
done:
    return code;
}

/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_fill_parallelogram(gx_device *dev, fixed px, fixed py, fixed ax, fixed ay, 
                       fixed bx, fixed by, const gx_drawing_color *pdcolor, 
                       gs_logical_operation_t lop)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;
    int code;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_fill_parallelogram() ...\n");
#endif

    if ( pngbestdev->color_count_is_accurate && pngbestdev->color_count <= 256 )
    {
        if ( pngbest_capture_color(dev, pdcolor->colors.pure) )
        {
            code = gs_note_error(gs_error_VMerror);
            goto done;
        }
    }
    else
    {
        /* unregister this callback function */
        dev->procs.fill_parallelogram = pngbestdev->orig_fill_parallelogram;
    }

    code = pngbestdev->orig_fill_parallelogram(dev, px, py, ax, ay, bx, by, pdcolor, lop);
   
done:
    return code;
}

/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_fill_triangle(gx_device *dev, fixed px, fixed py, fixed ax, fixed ay, 
                  fixed bx, fixed by, const gx_drawing_color *pdcolor, 
                  gs_logical_operation_t lop)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;
    int code;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_fill_triangle() ...\n");
#endif    

    if ( pngbestdev->color_count_is_accurate && pngbestdev->color_count <= 256 )
    {
        if ( pngbest_capture_color(dev, pdcolor->colors.pure) )
        {
            code = gs_note_error(gs_error_VMerror);
            goto done;
        }
    }
    else
    {
        /* unregister this callback function */
        dev->procs.fill_triangle = pngbestdev->orig_fill_triangle;
    }

    code = pngbestdev->orig_fill_triangle(dev, px, py, ax, ay, bx, by, pdcolor, lop);
   
done:
    return code;
}

/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_draw_thin_line(gx_device * dev,
                   fixed fx0, fixed fy0, fixed fx1, fixed fy1,
                   const gx_device_color * pdcolor, gs_logical_operation_t lop,
				   fixed adjustx, fixed adjusty)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;
    int code;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_draw_thin_line() ...\n");
#endif    

    if ( pngbestdev->color_count_is_accurate && pngbestdev->color_count <= 256 )
    {
        if ( pngbest_capture_color(dev, pdcolor->colors.pure) )
        {
            code = gs_note_error(gs_error_VMerror);
            goto done;
        }
    }
    else
    {
        /* unregister this callback function */
        dev->procs.draw_thin_line = pngbestdev->orig_draw_thin_line;
    }

    code = pngbestdev->orig_draw_thin_line(dev, fx0, fy0, fx1, fy1, pdcolor, lop,
		                                   adjustx, adjusty);
   
done:
    return code;
}

/* --------------------------------------------------------------------------*/
/* --- not supported --------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_begin_image(gx_device *dev, const gs_imager_state *pis, 
                const gs_image_t *pim, gs_image_format_t format, 
                const gs_int_rect *prect, const gx_drawing_color *pdcolor, 
                const gx_clip_path *pcpath, gs_memory_t *memory, 
                gx_image_enum_common_t **pinfo)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_begin_image() ...\n");
#endif    

    /* we don't know the outcome of this operation */
    pngbestdev->color_count_is_accurate = 0;

    /* unregister this callback function */
    dev->procs.begin_image = pngbestdev->orig_begin_image;

#ifdef SHOW_CALLBACK_NAMES
    printf("\n\n!!!\n\n");
#endif

    return dev->procs.begin_image(dev, pis, pim, format, prect, 
                                  pdcolor, pcpath, memory, pinfo);
}

/* --------------------------------------------------------------------------*/
/* --- not supported --------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int 
pngbest_strip_tile_rectangle(gx_device *dev, const gx_strip_bitmap *tile, 
                         int x, int y, int width, int height, 
                         gx_color_index color0, gx_color_index color1, 
                         int phase_x, int phase_y)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_strip_tile_rectangle() ...\n");
#endif    

    /* we don't know the outcome of this operation */
    pngbestdev->color_count_is_accurate = 0;

    /* unregister this callback function */
    dev->procs.strip_tile_rectangle = pngbestdev->orig_strip_tile_rectangle;

#ifdef SHOW_CALLBACK_NAMES
    printf("\n\n!!!\n\n");
#endif

    return dev->procs.strip_tile_rectangle(dev, tile, x, y, width, height, 
                                           color0, color1, phase_x, phase_y);
}

/* --------------------------------------------------------------------------*/
/* --- not supported --------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_strip_copy_rop(gx_device *dev, const byte *sdata, int sourcex, 
                   uint sraster, gx_bitmap_id id, 
                   const gx_color_index *scolors, 
                   const gx_strip_bitmap *texture, 
                   const gx_color_index *tcolors, 
                   int x, int y, int width, int height, 
                   int phase_x, int phase_y, gs_logical_operation_t lop)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_strip_copy_rop() ...\n");
#endif    

    /* we don't know the outcome of this operation */
    pngbestdev->color_count_is_accurate = 0;

    /* unregister this callback function */
    dev->procs.strip_copy_rop = pngbestdev->orig_strip_copy_rop;

#ifdef SHOW_CALLBACK_NAMES
    printf("\n\n!!!\n\n");
#endif

    return dev->procs.strip_copy_rop(dev, sdata, sourcex, sraster, id, scolors, 
                                     texture, tcolors, x, y, width, height, 
                                     phase_x, phase_y, lop);
}

/* --------------------------------------------------------------------------*/
/* --- not supported --------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_begin_typed_image(gx_device *dev, const gs_imager_state *pis, 
                      const gs_matrix *pmat, const gs_image_common_t *pim, 
                      const gs_int_rect *prect, const gx_drawing_color *pdcolor,
                      const gx_clip_path *pcpath, gs_memory_t *memory, 
                      gx_image_enum_common_t **pinfo)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_begin_typed_image() ...\n");
#endif    

    /* we don't know the outcome of this operation */
    pngbestdev->color_count_is_accurate = 0;

    /* unregister this callback function */
    dev->procs.begin_typed_image = pngbestdev->orig_begin_typed_image;

#ifdef SHOW_CALLBACK_NAMES
    printf("\n\n!!!\n\n");
#endif

    return dev->procs.begin_typed_image(dev, pis, pmat, pim, prect, pdcolor, 
                                        pcpath, memory, pinfo);
}

/* --------------------------------------------------------------------------*/
/* --- not supported --------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_create_compositor(gx_device *dev, gx_device **pcdev,
                      const gs_composite_t *pcte,
                      gs_imager_state *pis, gs_memory_t *memory, 
                      gx_device *cindev)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_create_compositor() ...\n");
#endif

    /* we don't know the outcome of this operation */
    pngbestdev->color_count_is_accurate = 0;

    /* unregister this callback function */
    dev->procs.create_compositor = pngbestdev->orig_create_compositor;

#ifdef SHOW_CALLBACK_NAMES
    printf("\n\n!!!\n\n");
#endif

    return dev->procs.create_compositor(dev, pcdev, pcte, pis, memory, cindev);
}

/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/
/* --------------------------------------------------------------------------*/

static int
pngbest_text_begin(gx_device * dev, gs_imager_state * pis,
               const gs_text_params_t * text, gs_font * font,
               gx_path * path, const gx_device_color * pdcolor,
               const gx_clip_path * pcpath,
               gs_memory_t * memory, gs_text_enum_t ** ppenum)
{
    gx_device_pngbest *pngbestdev = (gx_device_pngbest *)dev;
    int code;

#ifdef SHOW_CALLBACK_NAMES
    printf("pngbest_text_begin() ...\n");
#endif

    if ( text->operation & TEXT_DO_DRAW )
    {
        /* track color count */
        if ( pngbestdev->color_count_is_accurate && pngbestdev->color_count <= 256 )
        {
            if ( pngbest_capture_color(dev, pdcolor->colors.pure) )
            {
                code = gs_note_error(gs_error_VMerror);
                goto done;
            }
        }

        /* capture text output */
        if ( !(text->size == 1 && text->data.bytes[0] == ' ') )
        {
            if ( pngbestdev->rotated == -1 )
                pngbest_check_rotation(pngbestdev);

        }
    }

    code = gx_default_text_begin(dev, pis, text, font, path, pdcolor,
                                 pcpath, memory, ppenum);

done:

    return code;
}
